
-- 
--  Anaglym std library
-- 
--  History:
--    2009-11-02  Josh Holtrop       Initial Revision
--

std = {}

--  Compute the cross product of two vectors
--  Input:
--    vec1: the first vector (table; indices 1..3)
--    vec2: the second vector (table; indices 1..3)
--  Output:
--    The cross product vector (table; indices 1..3)
std.crossProduct = function(vec1, vec2)
    local vec_result = {}
    vec_result[1] = vec1[2] * vec2[3] - vec1[3] * vec2[2]
    vec_result[2] = vec1[3] * vec2[1] - vec1[1] * vec2[3]
    vec_result[3] = vec1[1] * vec2[2] - vec1[2] * vec2[1]
    return vec_result
end

--  Compute the dot product of two vectors
--  Input:
--    vec1: the first vector (table; indices 1..3)
--    vec2: the second vector (table; indices 1..3)
--  Output:
--    The dot product scalar
std.dotProduct = function(vec1, vec2)
    return vec1[1] * vec2[1] + vec1[2] * vec2[2] + vec1[3] * vec2[3]
end

--  Create a plane
--  Input:
--    x, y, z: coordinates of a point on the plane
--    nx, ny, nz: normal vector of the plane
--  Output:
--    Plane object (returned from ag.createPlane())
std.createPlanePointNormal = function(x, y, z, nx, ny, nz)
    -- calculate the d plane parameter based on the point coordinates
    -- invoke the ag routine to create a plane based on a, b, c, d parameters
    return ag.createPlane(nx, ny, nz, nx * x, ny * y, nz * z)
end

-- Convert screen percent to pixel count along X axis
-- Input:
--   percent: percent (0.0 - 1.0) of the screen's X axis
-- Output:
--   Number of pixels corresponding to 'percent' percent of the X axis
std.pctx = function(percent)
    local width = ag.getScreenSize()
    return width * percent
end

-- Convert screen percent to pixel count along Y axis
-- Input:
--   percent: percent (0.0 - 1.0) of the screen's Y axis
-- Output:
--   Number of pixels corresponding to 'percent' percent of the Y axis
std.pcty = function(percent)
    local width, height = ag.getScreenSize()
    return height * percent
end

-- Load an object from a model file and figure out what scale
-- to use based on a maximum specified bounding box the model must fit in.
-- Input:
--   model_name: the string containing the name of the model to load
--   max_x, max_y, max_z: 
std.loadModelBounds = function(model_name, max_x, max_y, max_z)
    local tmp_model = ag.loadModel(model_name, {reference = true})
    local sx, sy, sz = tmp_model:getSize()
    tmp_model:destroy()
    local scale = 1.0
    local scale_set = false
    if (max_x > 0 and sx > 0) then
        scale = max_x / sx
        scale_set = true
    end
    if (max_y > 0 and sy > 0) then
        local s = max_y / sy
        if (s < scale or not scale_set) then
            scale = s
            scale_set = true
        end
    end
    if (max_z > 0 and sz > 0) then
        local s = max_z / sz
        if (s < scale or not scale_set) then
            scale = s
            scale_set = true
        end
    end
    return ag.loadModel(model_name, {scale = scale})
end
